<?php
/**
 * Plugin Settings Model.
 *
 * @package WPTravelEngine/Core/Models
 * @since 6.0.0
 */

namespace WPTravelEngine\Core\Models\Settings;

use WP_Term;
use WPTravelEngine\Traits\Factory;
use WPTravelEngine\Utilities\ArrayUtility;

/**
 * Class PluginSettings.
 *
 * @since 6.0.0
 */
class PluginSettings extends BaseSetting {

	use Factory;

	/**
	 * The callbacks to get the settings.
	 */
	protected array $callbacks;

	/**
	 * Constructor to set the option name and optional default settings.
	 */
	public function __construct() {
		parent::__construct( 'wp_travel_engine_settings', array() );

		/**
		 * Clear cached settings and reload fresh data from the database to ensure current values.
		 *
		 * @since 6.6.6
		 */
		$this->load_settings();

		$this->callbacks = array(
			'pages.wp_travel_engine_wishlist'              => array( $this, 'get_wishlist_page' ),
			'pages.search'                                 => array( $this, 'get_search_page' ),
			'trip_tabs'                                    => array( $this, 'get_trip_tabs' ),
			'default_trip_facts'                           => array( $this, 'get_default_trip_infos' ),
			'trip_facts'                                   => array( $this, 'get_trashable_trip_infos' ),
			'email.emails'                                 => array( $this, 'get_emails' ),
			'email.booking_notification_subject_admin'     => array( $this, 'get_admin_booking_notify_subject' ),
			'email.booking_notification_template_admin'    => array( $this, 'get_admin_booking_notify_template' ),
			'email.sale_subject'                           => array( $this, 'get_admin_payment_notify_subject' ),
			'email.sales_wpeditor'                         => array( $this, 'get_admin_payment_notify_template' ),
			'email.booking_notification_subject_customer'  => array( $this, 'get_customer_booking_notify_subject' ),
			'email.booking_notification_template_customer' => array( $this, 'get_customer_booking_notify_template' ),
			'email.subject'                                => array( $this, 'get_customer_purchase_notify_subject' ),
			'email.purchase_wpeditor'                      => array( $this, 'get_customer_purchase_notify_template' ),
		);
		$this->set_defaults();
	}

	/**
	 * Get the default settings.
	 *
	 * @param string|null $key The key of the default setting in dot-seperated path.
	 * @param mixed       $default_value
	 *
	 * @return mixed
	 * @since 6.2.0
	 */
	public function get( string $key = null, $default_value = null ) {

		if ( isset( $this->callbacks[ $key ] ) ) {
			$value = call_user_func( $this->callbacks[ $key ] );
		} elseif ( is_null( $key ) || wptravelengine_key_exists( parent::get(), explode( '.', $key ) ) ) {
				$value = parent::get( $key );
		} else {
			$this->set( $key, $default_value );
			$value = $default_value;
		}

		return $value;
	}

	/**
	 * Get wishlist page.
	 *
	 * @TODO: This is a temporary solution to set wishlist and trip search result page.
	 * @return int
	 */
	protected function get_wishlist_page() {

		$wishlist_page_id = parent::get( 'pages.wp_travel_engine_wishlist' );
		if ( is_null( $wishlist_page_id ) ) {
			$wishlist_page    = wptravelengine_get_page_by_title( 'Wishlist' );
			$wishlist_page_id = $wishlist_page->ID;
			$this->set( 'pages.wp_travel_engine_wishlist', $wishlist_page_id );
		}

		return $wishlist_page_id;
	}

	/**
	 * Get search page.
	 *
	 * @TODO: This is a temporary solution to set wishlist and trip search result page.
	 * @return int
	 */
	protected function get_search_page() {

		$search_page_id = parent::get( 'pages.search' );
		if ( is_null( $search_page_id ) ) {
			$search_page    = wptravelengine_get_page_by_title( 'Trip Search Result' );
			$search_page_id = $search_page->ID;
			$this->set( 'pages.search', $search_page_id );
		}

		return $search_page_id;
	}

	/**
	 * Get the trip tabs.
	 *
	 * @return array
	 * @since 6.2.0
	 */
	protected function get_trip_tabs(): array {

		$is_trip_tabs_set = ! is_null( parent::get( 'trip_tabs' ) );

		if ( $is_trip_tabs_set ) {
			$tabs = parent::get( 'trip_tabs' );
		} else {
			$tabs           = wte_get_default_settings_tab();
			$tabs['enable'] = array_fill_keys( array_keys( $tabs['id'] ), 'yes' );
		}

		if ( ! empty( array_filter( $tabs['icon'] ?? (array) '', 'is_string' ) ) ) {
			$temp_arr = array();
			foreach (
				$tabs['icon'] ?? array_fill(
					1,
					count( $tabs['id'] ),
					array()
				) as $key => $val
			) {
				$temp_arr[ $key ] = array(
					'icon'     => $val['icon'] ?? ( is_string( $val ) ? $val : ( $val[0] ?? '' ) ),
					'view_box' => $val['view_box'] ?? '',
					'path'     => $val['path'] ?? '',
				);
			}
			$tabs['icon'] = $temp_arr;
		}

		if ( ! $is_trip_tabs_set || isset( $temp_arr ) ) {
			$this->set( 'trip_tabs', $tabs );
		}

		return $tabs;
	}


	/**
	 * Get the default trip facts.
	 *
	 * @return array
	 * @since 6.2.0
	 */
	protected function get_default_trip_infos(): array {

		$default_infos = parent::get( 'default_trip_facts' ) ?? wptravelengine_get_trip_facts_default_options();

		if ( ! empty( array_filter( array_column( $default_infos, 'field_icon' ), 'is_string' ) ) ) {

			$def_icons = array(
				'fas fa-child'         => array(
					'icon'     => 'fas fa-child',
					'view_box' => '0 0 320 512',
					'path'     => 'M96 64a64 64 0 1 1 128 0A64 64 0 1 1 96 64zm48 320l0 96c0 17.7-14.3 32-32 32s-32-14.3-32-32l0-192.2L59.1 321c-9.4 15-29.2 19.4-44.1 10S-4.5 301.9 4.9 287l39.9-63.3C69.7 184 113.2 160 160 160s90.3 24 115.2 63.6L315.1 287c9.4 15 4.9 34.7-10 44.1s-34.7 4.9-44.1-10L240 287.8 240 480c0 17.7-14.3 32-32 32s-32-14.3-32-32l0-96-32 0z',
				),
				'fas fa-male'          => array(
					'icon'     => 'fas fa-person',
					'view_box' => '0 0 320 512',
					'path'     => 'M112 48a48 48 0 1 1 96 0 48 48 0 1 1 -96 0zm40 304l0 128c0 17.7-14.3 32-32 32s-32-14.3-32-32l0-223.1L59.4 304.5c-9.1 15.1-28.8 20-43.9 10.9s-20-28.8-10.9-43.9l58.3-97c17.4-28.9 48.6-46.6 82.3-46.6l29.7 0c33.7 0 64.9 17.7 82.3 46.6l58.3 97c9.1 15.1 4.2 34.8-10.9 43.9s-34.8 4.2-43.9-10.9L232 256.9 232 480c0 17.7-14.3 32-32 32s-32-14.3-32-32l0-128-16 0z',
				),
				'fas fa-user-group'    => array(
					'icon'     => 'fas fa-user-group',
					'view_box' => '0 0 640 512',
					'path'     => 'M96 128a128 128 0 1 1 256 0A128 128 0 1 1 96 128zM0 482.3C0 383.8 79.8 304 178.3 304l91.4 0C368.2 304 448 383.8 448 482.3c0 16.4-13.3 29.7-29.7 29.7L29.7 512C13.3 512 0 498.7 0 482.3zM609.3 512l-137.8 0c5.4-9.4 8.6-20.3 8.6-32l0-8c0-60.7-27.1-115.2-69.8-151.8c2.4-.1 4.7-.2 7.1-.2l61.4 0C567.8 320 640 392.2 640 481.3c0 17-13.8 30.7-30.7 30.7zM432 256c-31 0-59-12.6-79.3-32.9C372.4 196.5 384 163.6 384 128c0-26.8-6.6-52.1-18.3-74.3C384.3 40.1 407.2 32 432 32c61.9 0 112 50.1 112 112s-50.1 112-112 112z',
				),
				'fas fa-shoe-prints'   => array(
					'icon'     => 'fas fa-shoe-prints',
					'view_box' => '0 0 640 512',
					'path'     => 'M416 0C352.3 0 256 32 256 32l0 128c48 0 76 16 104 32s56 32 104 32c56.4 0 176-16 176-96S512 0 416 0zM128 96c0 35.3 28.7 64 64 64l32 0 0-128-32 0c-35.3 0-64 28.7-64 64zM288 512c96 0 224-48 224-128s-119.6-96-176-96c-48 0-76 16-104 32s-56 32-104 32l0 128s96.3 32 160 32zM0 416c0 35.3 28.7 64 64 64l32 0 0-128-32 0c-35.3 0-64 28.7-64 64z',
				),
				'fas fa-person-hiking' => array(
					'icon'     => 'fas fa-person-hiking',
					'view_box' => '0 0 384 512',
					'path'     => 'M192 48a48 48 0 1 1 96 0 48 48 0 1 1 -96 0zm51.3 182.7L224.2 307l49.7 49.7c9 9 14.1 21.2 14.1 33.9l0 89.4c0 17.7-14.3 32-32 32s-32-14.3-32-32l0-82.7-73.9-73.9c-15.8-15.8-22.2-38.6-16.9-60.3l20.4-84c8.3-34.1 42.7-54.9 76.7-46.4c19 4.8 35.6 16.4 46.4 32.7L305.1 208l30.9 0 0-24c0-13.3 10.7-24 24-24s24 10.7 24 24l0 55.8c0 .1 0 .2 0 .2s0 .2 0 .2L384 488c0 13.3-10.7 24-24 24s-24-10.7-24-24l0-216-39.4 0c-16 0-31-8-39.9-21.4l-13.3-20zM81.1 471.9L117.3 334c3 4.2 6.4 8.2 10.1 11.9l41.9 41.9L142.9 488.1c-4.5 17.1-22 27.3-39.1 22.8s-27.3-22-22.8-39.1zm55.5-346L101.4 266.5c-3 12.1-14.9 19.9-27.2 17.9l-47.9-8c-14-2.3-22.9-16.3-19.2-30L31.9 155c9.5-34.8 41.1-59 77.2-59l4.2 0c15.6 0 27.1 14.7 23.3 29.8z',
				),
			);

			foreach ( $default_infos as $key => $value ) {
				$icon = $def_icons[ $value['field_icon'] ] ?? $value['field_icon'] ?? '';
				if ( is_array( $icon ) ) {
					$value['field_icon'] = $icon;
				} else {
					$value['field_icon'] = array(
						'icon'     => $icon,
						'view_box' => '',
						'path'     => '',
					);
				}
				$default_infos[ $key ] = $value;
			}

			$this->set( 'default_trip_facts', $default_infos );
		}

		return $default_infos;
	}

	/**
	 * Get the trashable default trip tabs.
	 *
	 * @return array
	 * @since 6.2.0
	 */
	protected function get_trashable_trip_infos(): array {

		$trashable_infos = parent::get( 'trip_facts' ) ?? wptravelengine_get_trip_facts_options();

		$field_icons = $trashable_infos['field_icon'] ?? array();
		if ( ! empty( array_filter( $field_icons, 'is_string' ) ) || is_null( parent::get( 'trip_facts' ) ) ) {

			$trashable_arr = array(
				'fas fa-hotel'            => array(
					'icon'     => 'fas fa-hotel',
					'view_box' => '0 0 512 512',
					'path'     => 'M0 32C0 14.3 14.3 0 32 0L480 0c17.7 0 32 14.3 32 32s-14.3 32-32 32l0 384c17.7 0 32 14.3 32 32s-14.3 32-32 32l-176 0 0-48c0-26.5-21.5-48-48-48s-48 21.5-48 48l0 48L32 512c-17.7 0-32-14.3-32-32s14.3-32 32-32L32 64C14.3 64 0 49.7 0 32zm96 80l0 32c0 8.8 7.2 16 16 16l32 0c8.8 0 16-7.2 16-16l0-32c0-8.8-7.2-16-16-16l-32 0c-8.8 0-16 7.2-16 16zM240 96c-8.8 0-16 7.2-16 16l0 32c0 8.8 7.2 16 16 16l32 0c8.8 0 16-7.2 16-16l0-32c0-8.8-7.2-16-16-16l-32 0zm112 16l0 32c0 8.8 7.2 16 16 16l32 0c8.8 0 16-7.2 16-16l0-32c0-8.8-7.2-16-16-16l-32 0c-8.8 0-16 7.2-16 16zM112 192c-8.8 0-16 7.2-16 16l0 32c0 8.8 7.2 16 16 16l32 0c8.8 0 16-7.2 16-16l0-32c0-8.8-7.2-16-16-16l-32 0zm112 16l0 32c0 8.8 7.2 16 16 16l32 0c8.8 0 16-7.2 16-16l0-32c0-8.8-7.2-16-16-16l-32 0c-8.8 0-16 7.2-16 16zm144-16c-8.8 0-16 7.2-16 16l0 32c0 8.8 7.2 16 16 16l32 0c8.8 0 16-7.2 16-16l0-32c0-8.8-7.2-16-16-16l-32 0zM328 384c13.3 0 24.3-10.9 21-23.8c-10.6-41.5-48.2-72.2-93-72.2s-82.5 30.7-93 72.2c-3.3 12.8 7.8 23.8 21 23.8l144 0z',
				),
				'fas fa-tag'              => array(
					'icon'     => 'fas fa-tag',
					'view_box' => '0 0 448 512',
					'path'     => 'M0 80L0 229.5c0 17 6.7 33.3 18.7 45.3l176 176c25 25 65.5 25 90.5 0L418.7 317.3c25-25 25-65.5 0-90.5l-176-176c-12-12-28.3-18.7-45.3-18.7L48 32C21.5 32 0 53.5 0 80zm112 32a32 32 0 1 1 0 64 32 32 0 1 1 0-64z',
				),
				'fas fa-city'             => array(
					'icon'     => 'fas fa-city',
					'view_box' => '0 0 640 512',
					'path'     => 'M480 48c0-26.5-21.5-48-48-48L336 0c-26.5 0-48 21.5-48 48l0 48-64 0 0-72c0-13.3-10.7-24-24-24s-24 10.7-24 24l0 72-64 0 0-72c0-13.3-10.7-24-24-24S64 10.7 64 24l0 72L48 96C21.5 96 0 117.5 0 144l0 96L0 464c0 26.5 21.5 48 48 48l256 0 32 0 96 0 160 0c26.5 0 48-21.5 48-48l0-224c0-26.5-21.5-48-48-48l-112 0 0-144zm96 320l0 32c0 8.8-7.2 16-16 16l-32 0c-8.8 0-16-7.2-16-16l0-32c0-8.8 7.2-16 16-16l32 0c8.8 0 16 7.2 16 16zM240 416l-32 0c-8.8 0-16-7.2-16-16l0-32c0-8.8 7.2-16 16-16l32 0c8.8 0 16 7.2 16 16l0 32c0 8.8-7.2 16-16 16zM128 400c0 8.8-7.2 16-16 16l-32 0c-8.8 0-16-7.2-16-16l0-32c0-8.8 7.2-16 16-16l32 0c8.8 0 16 7.2 16 16l0 32zM560 256c8.8 0 16 7.2 16 16l0 32c0 8.8-7.2 16-16 16l-32 0c-8.8 0-16-7.2-16-16l0-32c0-8.8 7.2-16 16-16l32 0zM256 176l0 32c0 8.8-7.2 16-16 16l-32 0c-8.8 0-16-7.2-16-16l0-32c0-8.8 7.2-16 16-16l32 0c8.8 0 16 7.2 16 16zM112 160c8.8 0 16 7.2 16 16l0 32c0 8.8-7.2 16-16 16l-32 0c-8.8 0-16-7.2-16-16l0-32c0-8.8 7.2-16 16-16l32 0zM256 304c0 8.8-7.2 16-16 16l-32 0c-8.8 0-16-7.2-16-16l0-32c0-8.8 7.2-16 16-16l32 0c8.8 0 16 7.2 16 16l0 32zM112 320l-32 0c-8.8 0-16-7.2-16-16l0-32c0-8.8 7.2-16 16-16l32 0c8.8 0 16 7.2 16 16l0 32c0 8.8-7.2 16-16 16zm304-48l0 32c0 8.8-7.2 16-16 16l-32 0c-8.8 0-16-7.2-16-16l0-32c0-8.8 7.2-16 16-16l32 0c8.8 0 16 7.2 16 16zM400 64c8.8 0 16 7.2 16 16l0 32c0 8.8-7.2 16-16 16l-32 0c-8.8 0-16-7.2-16-16l0-32c0-8.8 7.2-16 16-16l32 0zm16 112l0 32c0 8.8-7.2 16-16 16l-32 0c-8.8 0-16-7.2-16-16l0-32c0-8.8 7.2-16 16-16l32 0c8.8 0 16 7.2 16 16z',
				),
				'fas fa-cloud-sun'        => array(
					'icon'     => 'fas fa-cloud-sun',
					'view_box' => '0 0 640 512',
					'path'     => 'M294.2 1.2c5.1 2.1 8.7 6.7 9.6 12.1l14.1 84.7 84.7 14.1c5.4 .9 10 4.5 12.1 9.6s1.5 10.9-1.6 15.4l-38.5 55c-2.2-.1-4.4-.2-6.7-.2c-23.3 0-45.1 6.2-64 17.1l0-1.1c0-53-43-96-96-96s-96 43-96 96s43 96 96 96c8.1 0 15.9-1 23.4-2.9c-36.6 18.1-63.3 53.1-69.8 94.9l-24.4 17c-4.5 3.2-10.3 3.8-15.4 1.6s-8.7-6.7-9.6-12.1L98.1 317.9 13.4 303.8c-5.4-.9-10-4.5-12.1-9.6s-1.5-10.9 1.6-15.4L52.5 208 2.9 137.2c-3.2-4.5-3.8-10.3-1.6-15.4s6.7-8.7 12.1-9.6L98.1 98.1l14.1-84.7c.9-5.4 4.5-10 9.6-12.1s10.9-1.5 15.4 1.6L208 52.5 278.8 2.9c4.5-3.2 10.3-3.8 15.4-1.6zM144 208a64 64 0 1 1 128 0 64 64 0 1 1 -128 0zM639.9 431.9c0 44.2-35.8 80-80 80l-271.9 0c-53 0-96-43-96-96c0-47.6 34.6-87 80-94.6l0-1.3c0-53 43-96 96-96c34.9 0 65.4 18.6 82.2 46.4c13-9.1 28.8-14.4 45.8-14.4c44.2 0 80 35.8 80 80c0 5.9-.6 11.7-1.9 17.2c37.4 6.7 65.8 39.4 65.8 78.7z',
				),
				'fas fa-sun-plant-wilt'   => array(
					'icon'     => 'fas fa-sun-plant-wilt',
					'view_box' => '0 0 640 512',
					'path'     => 'M160 0c-6.3 0-12 3.7-14.6 9.5L120.6 64.9 63.9 43.2c-5.9-2.3-12.6-.8-17 3.6s-5.9 11.1-3.6 17l21.7 56.7L9.5 145.4C3.7 148 0 153.7 0 160s3.7 12 9.5 14.6l55.4 24.8L43.2 256.1c-2.3 5.9-.8 12.6 3.6 17s11.1 5.9 17 3.6l56.7-21.7 24.8 55.4c2.6 5.8 8.3 9.5 14.6 9.5s12-3.7 14.6-9.5l24.8-55.4 56.7 21.7c5.9 2.3 12.6 .8 17-3.6s5.9-11.1 3.6-17l-21.7-56.7 55.4-24.8c5.8-2.6 9.5-8.3 9.5-14.6s-3.7-12-9.5-14.6l-55.4-24.8 21.7-56.7c2.3-5.9 .8-12.6-3.6-17s-11.1-5.9-17-3.6L199.4 64.9 174.6 9.5C172 3.7 166.3 0 160 0zm0 96a64 64 0 1 1 0 128 64 64 0 1 1 0-128zm32 64a32 32 0 1 0 -64 0 32 32 0 1 0 64 0zm312 16c0-17.7 14.3-32 32-32s32 14.3 32 32l0 53.4c-14.8 7.7-24 23.1-24 44.6c0 16.8 16 44 37.4 67.2c5.8 6.2 15.5 6.2 21.2 0C624 318 640 290.7 640 274c0-21.5-9.2-37-24-44.6l0-53.4c0-44.2-35.8-80-80-80s-80 35.8-80 80l0 22.7c-9.8-4.3-20.6-6.7-32-6.7c-44.2 0-80 35.8-80 80l0 21.4c-14.8 7.7-24 23.1-24 44.6c0 16.8 16 44 37.4 67.2c5.8 6.2 15.5 6.2 21.2 0C400 382 416 354.7 416 338c0-21.5-9.2-37-24-44.6l0-21.4c0-17.7 14.3-32 32-32s32 14.3 32 32l0 8 0 168L32 448c-17.7 0-32 14.3-32 32s14.3 32 32 32l576 0c17.7 0 32-14.3 32-32s-14.3-32-32-32l-104 0 0-168 0-8 0-96z	',
				),
				'far fa-calendar-xmark'   => array(
					'icon'     => 'far fa-calendar-xmark',
					'view_box' => '0 0 448 512',
					'path'     => 'M128 0c17.7 0 32 14.3 32 32l0 32 128 0 0-32c0-17.7 14.3-32 32-32s32 14.3 32 32l0 32 48 0c26.5 0 48 21.5 48 48l0 48L0 160l0-48C0 85.5 21.5 64 48 64l48 0 0-32c0-17.7 14.3-32 32-32zM0 192l448 0 0 272c0 26.5-21.5 48-48 48L48 512c-26.5 0-48-21.5-48-48L0 192zM305 305c9.4-9.4 9.4-24.6 0-33.9s-24.6-9.4-33.9 0l-47 47-47-47c-9.4-9.4-24.6-9.4-33.9 0s-9.4 24.6 0 33.9l47 47-47 47c-9.4 9.4-9.4 24.6 0 33.9s24.6 9.4 33.9 0l47-47 47 47c9.4 9.4 24.6 9.4 33.9 0s9.4-24.6 0-33.9l-47-47 47-47z',
				),
				'fas fa-hands-praying'    => array(
					'icon'     => 'fas fa-hands-praying',
					'view_box' => '0 0 640 512',
					'path'     => 'M320 0c-88.2 0-160 71.8-160 160c0 88.2 71.8 160 160 160c88.2 0 160-71.8 160-160C480 71.8 408.2 0 320 0zM320 256c-35.3 0-64-28.7-64-64c0-35.3 28.7-64 64-64c35.3 0 64 28.7 64 64C384 227.3 355.3 256 320 256zM320 96c-26.5 0-48 21.5-48 48c0 26.5 21.5 48 48 48c26.5 0 48-21.5 48-48C368 117.5 346.5 96 320 96zM320 416c-26.5 0-48 21.5-48 48c0 26.5 21.5 48 48 48c26.5 0 48-21.5 48-48C368 437.5 346.5 416 320 416z',
				),
				'fas fa-hospital-user'    => array(
					'icon'     => 'fas fa-hospital-user',
					'view_box' => '0 0 576 512',
					'path'     => 'M48 0C21.5 0 0 21.5 0 48L0 256l144 0c8.8 0 16 7.2 16 16s-7.2 16-16 16L0 288l0 64 144 0c8.8 0 16 7.2 16 16s-7.2 16-16 16L0 384l0 80c0 26.5 21.5 48 48 48l217.9 0c-6.3-10.2-9.9-22.2-9.9-35.1c0-46.9 25.8-87.8 64-109.2l0-95.9L320 48c0-26.5-21.5-48-48-48L48 0zM152 64l16 0c8.8 0 16 7.2 16 16l0 24 24 0c8.8 0 16 7.2 16 16l0 16c0 8.8-7.2 16-16 16l-24 0 0 24c0 8.8-7.2 16-16 16l-16 0c-8.8 0-16-7.2-16-16l0-24-24 0c-8.8 0-16-7.2-16-16l0-16c0-8.8 7.2-16 16-16l24 0 0-24c0-8.8 7.2-16 16-16zM512 272a80 80 0 1 0 -160 0 80 80 0 1 0 160 0zM288 477.1c0 19.3 15.6 34.9 34.9 34.9l218.2 0c19.3 0 34.9-15.6 34.9-34.9c0-51.4-41.7-93.1-93.1-93.1l-101.8 0c-51.4 0-93.1 41.7-93.1 93.1z',
				),
				'fas fa-language'         => array(
					'icon'     => 'fas fa-language',
					'view_box' => '0 0 640 512',
					'path'     => 'M0 128C0 92.7 28.7 64 64 64l192 0 48 0 16 0 256 0c35.3 0 64 28.7 64 64l0 256c0 35.3-28.7 64-64 64l-256 0-16 0-48 0L64 448c-35.3 0-64-28.7-64-64L0 128zm320 0l0 256 256 0 0-256-256 0zM178.3 175.9c-3.2-7.2-10.4-11.9-18.3-11.9s-15.1 4.7-18.3 11.9l-64 144c-4.5 10.1 .1 21.9 10.2 26.4s21.9-.1 26.4-10.2l8.9-20.1 73.6 0 8.9 20.1c4.5 10.1 16.3 14.6 26.4 10.2s14.6-16.3 10.2-26.4l-64-144zM160 233.2L179 276l-38 0 19-42.8zM448 164c11 0 20 9 20 20l0 4 44 0 16 0c11 0 20 9 20 20s-9 20-20 20l-2 0-1.6 4.5c-8.9 24.4-22.4 46.6-39.6 65.4c.9 .6 1.8 1.1 2.7 1.6l18.9 11.3c9.5 5.7 12.5 18 6.9 27.4s-18 12.5-27.4 6.9l-18.9-11.3c-4.5-2.7-8.8-5.5-13.1-8.5c-10.6 7.5-21.9 14-34 19.4l-3.6 1.6c-10.1 4.5-21.9-.1-26.4-10.2s.1-21.9 10.2-26.4l3.6-1.6c6.4-2.9 12.6-6.1 18.5-9.8l-12.2-12.2c-7.8-7.8-7.8-20.5 0-28.3s20.5-7.8 28.3 0l14.6 14.6 .5 .5c12.4-13.1 22.5-28.3 29.8-45L448 228l-72 0c-11 0-20-9-20-20s9-20 20-20l52 0 0-4c0-11 9-20 20-20z',
				),
				'fas fa-mountain'         => array(
					'icon'     => 'fas fa-mountain',
					'view_box' => '0 0 640 512',
					'path'     => 'M256 32c12.5 0 24.1 6.4 30.8 17L503.4 394.4c5.6 8.9 8.6 19.2 8.6 29.7c0 30.9-25 55.9-55.9 55.9L55.9 480C25 480 0 455 0 424.1c0-10.5 3-20.8 8.6-29.7L225.2 49c6.6-10.6 18.3-17 30.8-17zm65 192L256 120.4 176.9 246.5l18.3 24.4c6.4 8.5 19.2 8.5 25.6 0l25.6-34.1c6-8.1 15.5-12.8 25.6-12.8l49 0z',
				),
				'fas fa-bowl-food'        => array(
					'icon'     => 'fas fa-bowl-food',
					'view_box' => '0 0 640 512',
					'path'     => 'M0 192c0-35.3 28.7-64 64-64c.5 0 1.1 0 1.6 0C73 91.5 105.3 64 144 64c15 0 29 4.1 40.9 11.2C198.2 49.6 225.1 32 256 32s57.8 17.6 71.1 43.2C339 68.1 353 64 368 64c38.7 0 71 27.5 78.4 64c.5 0 1.1 0 1.6 0c35.3 0 64 28.7 64 64c0 11.7-3.1 22.6-8.6 32L8.6 224C3.1 214.6 0 203.7 0 192zm0 91.4C0 268.3 12.3 256 27.4 256l457.1 0c15.1 0 27.4 12.3 27.4 27.4c0 70.5-44.4 130.7-106.7 154.1L403.5 452c-2 16-15.6 28-31.8 28l-231.5 0c-16.1 0-29.8-12-31.8-28l-1.8-14.4C44.4 414.1 0 353.9 0 283.4z',
				),
				'fas fa-handshake-simple' => array(
					'icon'     => 'fas fa-handshake-simple',
					'view_box' => '0 0 640 512',
					'path'     => 'M323.4 85.2l-96.8 78.4c-16.1 13-19.2 36.4-7 53.1c12.9 17.8 38 21.3 55.3 7.8l99.3-77.2c7-5.4 17-4.2 22.5 2.8s4.2 17-2.8 22.5l-20.9 16.2L550.2 352l41.8 0c26.5 0 48-21.5 48-48l0-128c0-26.5-21.5-48-48-48l-76 0-4 0-.7 0-3.9-2.5L434.8 79c-15.3-9.8-33.2-15-51.4-15c-21.8 0-43 7.5-60 21.2zm22.8 124.4l-51.7 40.2C263 274.4 217.3 268 193.7 235.6c-22.2-30.5-16.6-73.1 12.7-96.8l83.2-67.3c-11.6-4.9-24.1-7.4-36.8-7.4C234 64 215.7 69.6 200 80l-72 48-80 0c-26.5 0-48 21.5-48 48L0 304c0 26.5 21.5 48 48 48l108.2 0 91.4 83.4c19.6 17.9 49.9 16.5 67.8-3.1c5.5-6.1 9.2-13.2 11.1-20.6l17 15.6c19.5 17.9 49.9 16.6 67.8-2.9c4.5-4.9 7.8-10.6 9.9-16.5c19.4 13 45.8 10.3 62.1-7.5c17.9-19.5 16.6-49.9-2.9-67.8l-134.2-123z',
				),
				'fas fa-bottle-droplet'   => array(
					'icon'     => 'fas fa-bottle-droplet',
					'view_box' => '0 0 320 512',
					'path'     => 'M96 0C82.7 0 72 10.7 72 24s10.7 24 24 24c4.4 0 8 3.6 8 8l0 64.9c0 12.2-7.2 23.1-17.2 30.1C53.7 174.1 32 212.5 32 256l0 192c0 35.3 28.7 64 64 64l128 0c35.3 0 64-28.7 64-64l0-192c0-43.5-21.7-81.9-54.8-105c-10-7-17.2-17.9-17.2-30.1L216 56c0-4.4 3.6-8 8-8c13.3 0 24-10.7 24-24s-10.7-24-24-24l-8 0s0 0 0 0s0 0 0 0L104 0s0 0 0 0s0 0 0 0L96 0zm64 382c-26.5 0-48-20.1-48-45c0-16.8 22.1-48.1 36.3-66.4c6-7.8 17.5-7.8 23.5 0C185.9 288.9 208 320.2 208 337c0 24.9-21.5 45-48 45z',
				),
				'fab fa-cc-mastercard'    => array(
					'icon'     => 'fab fa-cc-mastercard',
					'view_box' => '0 0 576 512',
					'path'     => 'M482.9 410.3c0 6.8-4.6 11.7-11.2 11.7-6.8 0-11.2-5.2-11.2-11.7 0-6.5 4.4-11.7 11.2-11.7 6.6 0 11.2 5.2 11.2 11.7zm-310.8-11.7c-7.1 0-11.2 5.2-11.2 11.7 0 6.5 4.1 11.7 11.2 11.7 6.5 0 10.9-4.9 10.9-11.7-.1-6.5-4.4-11.7-10.9-11.7zm117.5-.3c-5.4 0-8.7 3.5-9.5 8.7h19.1c-.9-5.7-4.4-8.7-9.6-8.7zm107.8 .3c-6.8 0-10.9 5.2-10.9 11.7 0 6.5 4.1 11.7 10.9 11.7 6.8 0 11.2-4.9 11.2-11.7 0-6.5-4.4-11.7-11.2-11.7zm105.9 26.1c0 .3 .3 .5 .3 1.1 0 .3-.3 .5-.3 1.1-.3 .3-.3 .5-.5 .8-.3 .3-.5 .5-1.1 .5-.3 .3-.5 .3-1.1 .3-.3 0-.5 0-1.1-.3-.3 0-.5-.3-.8-.5-.3-.3-.5-.5-.5-.8-.3-.5-.3-.8-.3-1.1 0-.5 0-.8 .3-1.1 0-.5 .3-.8 .5-1.1 .3-.3 .5-.3 .8-.5 .5-.3 .8-.3 1.1-.3 .5 0 .8 0 1.1 .3 .5 .3 .8 .3 1.1 .5s.2 .6 .5 1.1zm-2.2 1.4c.5 0 .5-.3 .8-.3 .3-.3 .3-.5 .3-.8 0-.3 0-.5-.3-.8-.3 0-.5-.3-1.1-.3h-1.6v3.5h.8V426h.3l1.1 1.4h.8l-1.1-1.3zM576 81v352c0 26.5-21.5 48-48 48H48c-26.5 0-48-21.5-48-48V81c0-26.5 21.5-48 48-48h480c26.5 0 48 21.5 48 48zM64 220.6c0 76.5 62.1 138.5 138.5 138.5 27.2 0 53.9-8.2 76.5-23.1-72.9-59.3-72.4-171.2 0-230.5-22.6-15-49.3-23.1-76.5-23.1-76.4-.1-138.5 62-138.5 138.2zm224 108.8c70.5-55 70.2-162.2 0-217.5-70.2 55.3-70.5 162.6 0 217.5zm-142.3 76.3c0-8.7-5.7-14.4-14.7-14.7-4.6 0-9.5 1.4-12.8 6.5-2.4-4.1-6.5-6.5-12.2-6.5-3.8 0-7.6 1.4-10.6 5.4V392h-8.2v36.7h8.2c0-18.9-2.5-30.2 9-30.2 10.2 0 8.2 10.2 8.2 30.2h7.9c0-18.3-2.5-30.2 9-30.2 10.2 0 8.2 10 8.2 30.2h8.2v-23zm44.9-13.7h-7.9v4.4c-2.7-3.3-6.5-5.4-11.7-5.4-10.3 0-18.2 8.2-18.2 19.3 0 11.2 7.9 19.3 18.2 19.3 5.2 0 9-1.9 11.7-5.4v4.6h7.9V392zm40.5 25.6c0-15-22.9-8.2-22.9-15.2 0-5.7 11.9-4.8 18.5-1.1l3.3-6.5c-9.4-6.1-30.2-6-30.2 8.2 0 14.3 22.9 8.3 22.9 15 0 6.3-13.5 5.8-20.7 .8l-3.5 6.3c11.2 7.6 32.6 6 32.6-7.5zm35.4 9.3l-2.2-6.8c-3.8 2.1-12.2 4.4-12.2-4.1v-16.6h13.1V392h-13.1v-11.2h-8.2V392h-7.6v7.3h7.6V416c0 17.6 17.3 14.4 22.6 10.9zm13.3-13.4h27.5c0-16.2-7.4-22.6-17.4-22.6-10.6 0-18.2 7.9-18.2 19.3 0 20.5 22.6 23.9 33.8 14.2l-3.8-6c-7.8 6.4-19.6 5.8-21.9-4.9zm59.1-21.5c-4.6-2-11.6-1.8-15.2 4.4V392h-8.2v36.7h8.2V408c0-11.6 9.5-10.1 12.8-8.4l2.4-7.6zm10.6 18.3c0-11.4 11.6-15.1 20.7-8.4l3.8-6.5c-11.6-9.1-32.7-4.1-32.7 15 0 19.8 22.4 23.8 32.7 15l-3.8-6.5c-9.2 6.5-20.7 2.6-20.7-8.6zm66.7-18.3H408v4.4c-8.3-11-29.9-4.8-29.9 13.9 0 19.2 22.4 24.7 29.9 13.9v4.6h8.2V392zm33.7 0c-2.4-1.2-11-2.9-15.2 4.4V392h-7.9v36.7h7.9V408c0-11 9-10.3 12.8-8.4l2.4-7.6zm40.3-14.9h-7.9v19.3c-8.2-10.9-29.9-5.1-29.9 13.9 0 19.4 22.5 24.6 29.9 13.9v4.6h7.9v-51.7zm7.6-75.1v4.6h.8V302h1.9v-.8h-4.6v.8h1.9zm6.6 123.8c0-.5 0-1.1-.3-1.6-.3-.3-.5-.8-.8-1.1-.3-.3-.8-.5-1.1-.8-.5 0-1.1-.3-1.6-.3-.3 0-.8 .3-1.4 .3-.5 .3-.8 .5-1.1 .8-.5 .3-.8 .8-.8 1.1-.3 .5-.3 1.1-.3 1.6 0 .3 0 .8 .3 1.4 0 .3 .3 .8 .8 1.1 .3 .3 .5 .5 1.1 .8 .5 .3 1.1 .3 1.4 .3 .5 0 1.1 0 1.6-.3 .3-.3 .8-.5 1.1-.8 .3-.3 .5-.8 .8-1.1 .3-.6 .3-1.1 .3-1.4zm3.2-124.7h-1.4l-1.6 3.5-1.6-3.5h-1.4v5.4h.8v-4.1l1.6 3.5h1.1l1.4-3.5v4.1h1.1v-5.4zm4.4-80.5c0-76.2-62.1-138.3-138.5-138.3-27.2 0-53.9 8.2-76.5 23.1 72.1 59.3 73.2 171.5 0 230.5 22.6 15 49.5 23.1 76.5 23.1 76.4 .1 138.5-61.9 138.5-138.4z',
				),
				'fas fa-person-hiking'    => array(
					'icon'     => 'fas fa-person-hiking',
					'view_box' => '0 0 384 512',
					'path'     => 'M192 48a48 48 0 1 1 96 0 48 48 0 1 1 -96 0zm51.3 182.7L224.2 307l49.7 49.7c9 9 14.1 21.2 14.1 33.9l0 89.4c0 17.7-14.3 32-32 32s-32-14.3-32-32l0-82.7-73.9-73.9c-15.8-15.8-22.2-38.6-16.9-60.3l20.4-84c8.3-34.1 42.7-54.9 76.7-46.4c19 4.8 35.6 16.4 46.4 32.7L305.1 208l30.9 0 0-24c0-13.3 10.7-24 24-24s24 10.7 24 24l0 55.8c0 .1 0 .2 0 .2s0 .2 0 .2L384 488c0 13.3-10.7 24-24 24s-24-10.7-24-24l0-216-39.4 0c-16 0-31-8-39.9-21.4l-13.3-20zM81.1 471.9L117.3 334c3 4.2 6.4 8.2 10.1 11.9l41.9 41.9L142.9 488.1c-4.5 17.1-22 27.3-39.1 22.8s-27.3-22-22.8-39.1zm55.5-346L101.4 266.5c-3 12.1-14.9 19.9-27.2 17.9l-47.9-8c-14-2.3-22.9-16.3-19.2-30L31.9 155c9.5-34.8 41.1-59 77.2-59l4.2 0c15.6 0 27.1 14.7 23.3 29.8z',
				),
				'fas fa-bus'              => array(
					'icon'     => 'fas fa-bus',
					'view_box' => '0 0 576 512',
					'path'     => 'M288 0C422.4 0 512 35.2 512 80l0 16 0 32c17.7 0 32 14.3 32 32l0 64c0 17.7-14.3 32-32 32l0 160c0 17.7-14.3 32-32 32l0 32c0 17.7-14.3 32-32 32l-32 0c-17.7 0-32-14.3-32-32l0-32-192 0 0 32c0 17.7-14.3 32-32 32l-32 0c-17.7 0-32-14.3-32-32l0-32c-17.7 0-32-14.3-32-32l0-160c-17.7 0-32-14.3-32-32l0-64c0-17.7 14.3-32 32-32c0 0 0 0 0 0l0-32s0 0 0 0l0-16C64 35.2 153.6 0 288 0zM128 160l0 96c0 17.7 14.3 32 32 32l112 0 0-160-112 0c-17.7 0-32 14.3-32 32zM304 288l112 0c17.7 0 32-14.3 32-32l0-96c0-17.7-14.3-32-32-32l-112 0 0 160zM144 400a32 32 0 1 0 0-64 32 32 0 1 0 0 64zm288 0a32 32 0 1 0 0-64 32 32 0 1 0 0 64zM384 80c0-8.8-7.2-16-16-16L208 64c-8.8 0-16 7.2-16 16s7.2 16 16 16l160 0c8.8 0 16-7.2 16-16z',
				),
				'far fa-clock'            => array(
					'icon'     => 'far fa-clock',
					'view_box' => '0 0 512 512',
					'path'     => 'M464 256A208 208 0 1 1 48 256a208 208 0 1 1 416 0zM0 256a256 256 0 1 0 512 0A256 256 0 1 0 0 256zM232 120l0 136c0 8 4 15.5 10.7 20l96 64c11 7.4 25.9 4.4 33.3-6.7s4.4-25.9-6.7-33.3L280 243.2 280 120c0-13.3-10.7-24-24-24s-24 10.7-24 24z',
				),
				'fas fa-wifi'             => array(
					'icon'     => 'fas fa-wifi',
					'view_box' => '0 0 640 512',
					'path'     => 'M54.2 202.9C123.2 136.7 216.8 96 320 96s196.8 40.7 265.8 106.9c12.8 12.2 33 11.8 45.2-.9s11.8-33-.9-45.2C549.7 79.5 440.4 32 320 32S90.3 79.5 9.8 156.7C-2.9 169-3.3 189.2 8.9 202s32.5 13.2 45.2 .9zM320 256c56.8 0 108.6 21.1 148.2 56c13.3 11.7 33.5 10.4 45.2-2.8s10.4-33.5-2.8-45.2C459.8 219.2 393 192 320 192s-139.8 27.2-190.5 72c-13.3 11.7-14.5 31.9-2.8 45.2s31.9 14.5 45.2 2.8c39.5-34.9 91.3-56 148.2-56zm64 160a64 64 0 1 0 -128 0 64 64 0 1 0 128 0z',
				),
				// Demo Importer Icons
				'fas fa-people-group'     => array(
					'icon'     => 'fas fa-people-group',
					'view_box' => '0 0 640 512',
					'path'     => 'M72 88a56 56 0 1 1 112 0A56 56 0 1 1 72 88zM64 245.7C54 256.9 48 271.8 48 288s6 31.1 16 42.3l0-84.7zm144.4-49.3C178.7 222.7 160 261.2 160 304c0 34.3 12 65.8 32 90.5l0 21.5c0 17.7-14.3 32-32 32l-64 0c-17.7 0-32-14.3-32-32l0-26.8C26.2 371.2 0 332.7 0 288c0-61.9 50.1-112 112-112l32 0c24 0 46.2 7.5 64.4 20.3zM448 416l0-21.5c20-24.7 32-56.2 32-90.5c0-42.8-18.7-81.3-48.4-107.7C449.8 183.5 472 176 496 176l32 0c61.9 0 112 50.1 112 112c0 44.7-26.2 83.2-64 101.2l0 26.8c0 17.7-14.3 32-32 32l-64 0c-17.7 0-32-14.3-32-32zm8-328a56 56 0 1 1 112 0A56 56 0 1 1 456 88zM576 245.7l0 84.7c10-11.3 16-26.1 16-42.3s-6-31.1-16-42.3zM320 32a64 64 0 1 1 0 128 64 64 0 1 1 0-128zM240 304c0 16.2 6 31 16 42.3l0-84.7c-10 11.3-16 26.1-16 42.3zm144-42.3l0 84.7c10-11.3 16-26.1 16-42.3s-6-31.1-16-42.3zM448 304c0 44.7-26.2 83.2-64 101.2l0 42.8c0 17.7-14.3 32-32 32l-64 0c-17.7 0-32-14.3-32-32l0-42.8c-37.8-18-64-56.5-64-101.2c0-61.9 50.1-112 112-112l32 0c61.9 0 112 50.1 112 112z',
				),
				'fas fa-bed'              => array(
					'icon'     => 'fas fa-bed',
					'view_box' => '0 0 640 512',
					'path'     => 'M32 32c17.7 0 32 14.3 32 32l0 256 224 0 0-160c0-17.7 14.3-32 32-32l224 0c53 0 96 43 96 96l0 224c0 17.7-14.3 32-32 32s-32-14.3-32-32l0-32-224 0-32 0L64 416l0 32c0 17.7-14.3 32-32 32s-32-14.3-32-32L0 64C0 46.3 14.3 32 32 32zm144 96a80 80 0 1 1 0 160 80 80 0 1 1 0-160z',
				),
				'fas fa-plane-arrival'    => array(
					'icon'     => 'fas fa-plane-arrival',
					'view_box' => '0 0 640 512',
					'path'     => 'M.3 166.9L0 68C0 57.7 9.5 50.1 19.5 52.3l35.6 7.9c10.6 2.3 19.2 9.9 23 20L96 128l127.3 37.6L181.8 20.4C178.9 10.2 186.6 0 197.2 0l40.1 0c11.6 0 22.2 6.2 27.9 16.3l109 193.8 107.2 31.7c15.9 4.7 30.8 12.5 43.7 22.8l34.4 27.6c24 19.2 18.1 57.3-10.7 68.2c-41.2 15.6-86.2 18.1-128.8 7L121.7 289.8c-11.1-2.9-21.2-8.7-29.3-16.9L9.5 189.4c-5.9-6-9.3-14.1-9.3-22.5zM32 448l576 0c17.7 0 32 14.3 32 32s-14.3 32-32 32L32 512c-17.7 0-32-14.3-32-32s14.3-32 32-32zm96-80a32 32 0 1 1 64 0 32 32 0 1 1 -64 0zm128-16a32 32 0 1 1 0 64 32 32 0 1 1 0-64z',
				),
				'fas fa-plane-departure'  => array(
					'icon'     => 'fas fa-plane-departure',
					'view_box' => '0 0 640 512',
					'path'     => 'M381 114.9L186.1 41.8c-16.7-6.2-35.2-5.3-51.1 2.7L89.1 67.4C78 73 77.2 88.5 87.6 95.2l146.9 94.5L136 240 77.8 214.1c-8.7-3.9-18.8-3.7-27.3 .6L18.3 230.8c-9.3 4.7-11.8 16.8-5 24.7l73.1 85.3c6.1 7.1 15 11.2 24.3 11.2l137.7 0c5 0 9.9-1.2 14.3-3.4L535.6 212.2c46.5-23.3 82.5-63.3 100.8-112C645.9 75 627.2 48 600.2 48l-57.4 0c-20.2 0-40.2 4.8-58.2 14L381 114.9zM0 480c0 17.7 14.3 32 32 32l576 0c17.7 0 32-14.3 32-32s-14.3-32-32-32L32 448c-17.7 0-32 14.3-32 32z',
				),
				'fas fa-anchor'           => array(
					'icon'     => 'fas fa-anchor',
					'view_box' => '0 0 576 512',
					'path'     => 'M320 96a32 32 0 1 1 -64 0 32 32 0 1 1 64 0zm21.1 80C367 158.8 384 129.4 384 96c0-53-43-96-96-96s-96 43-96 96c0 33.4 17 62.8 42.9 80L224 176c-17.7 0-32 14.3-32 32s14.3 32 32 32l32 0 0 208-48 0c-53 0-96-43-96-96l0-6.1 7 7c9.4 9.4 24.6 9.4 33.9 0s9.4-24.6 0-33.9L97 263c-9.4-9.4-24.6-9.4-33.9 0L7 319c-9.4 9.4-9.4 24.6 0 33.9s24.6 9.4 33.9 0l7-7 0 6.1c0 88.4 71.6 160 160 160l80 0 80 0c88.4 0 160-71.6 160-160l0-6.1 7 7c9.4 9.4 24.6 9.4 33.9 0s9.4-24.6 0-33.9l-56-56c-9.4-9.4-24.6-9.4-33.9 0l-56 56c-9.4 9.4-9.4 24.6 0 33.9s24.6 9.4 33.9 0l7-7 0 6.1c0 53-43 96-96 96l-48 0 0-208 32 0c17.7 0 32-14.3 32-32s-14.3-32-32-32l-10.9 0z',
				),
				'fas fa-utensils'         => array(
					'icon'     => 'fas fa-utensils',
					'view_box' => '0 0 448 512',
					'path'     => 'M416 0C400 0 288 32 288 176l0 112c0 35.3 28.7 64 64 64l32 0 0 128c0 17.7 14.3 32 32 32s32-14.3 32-32l0-128 0-112 0-208c0-17.7-14.3-32-32-32zM64 16C64 7.8 57.9 1 49.7 .1S34.2 4.6 32.4 12.5L2.1 148.8C.7 155.1 0 161.5 0 167.9c0 45.9 35.1 83.6 80 87.7L80 480c0 17.7 14.3 32 32 32s32-14.3 32-32l0-224.4c44.9-4.1 80-41.8 80-87.7c0-6.4-.7-12.8-2.1-19.1L191.6 12.5c-1.8-8-9.3-13.3-17.4-12.4S160 7.8 160 16l0 134.2c0 5.4-4.4 9.8-9.8 9.8c-5.1 0-9.3-3.9-9.8-9L127.9 14.6C127.2 6.3 120.3 0 112 0s-15.2 6.3-15.9 14.6L83.7 151c-.5 5.1-4.7 9-9.8 9c-5.4 0-9.8-4.4-9.8-9.8L64 16zm48.3 152l-.3 0-.3 0 .3-.7 .3 .7z',
				),
				'far fa-address-card'     => array(
					'icon'     => 'far fa-address-card',
					'view_box' => '0 0 576 512',
					'path'     => 'M512 80c8.8 0 16 7.2 16 16l0 320c0 8.8-7.2 16-16 16L64 432c-8.8 0-16-7.2-16-16L48 96c0-8.8 7.2-16 16-16l448 0zM64 32C28.7 32 0 60.7 0 96L0 416c0 35.3 28.7 64 64 64l448 0c35.3 0 64-28.7 64-64l0-320c0-35.3-28.7-64-64-64L64 32zM208 256a64 64 0 1 0 0-128 64 64 0 1 0 0 128zm-32 32c-44.2 0-80 35.8-80 80c0 8.8 7.2 16 16 16l192 0c8.8 0 16-7.2 16-16c0-44.2-35.8-80-80-80l-64 0zM376 144c-13.3 0-24 10.7-24 24s10.7 24 24 24l80 0c13.3 0 24-10.7 24-24s-10.7-24-24-24l-80 0zm0 96c-13.3 0-24 10.7-24 24s10.7 24 24 24l80 0c13.3 0 24-10.7 24-24s-10.7-24-24-24l-80 0z',
				),
				'fas fa-hiking'           => array(
					'icon'     => 'fas fa-person-hiking',
					'view_box' => '0 0 384 512',
					'path'     => 'M192 48a48 48 0 1 1 96 0 48 48 0 1 1 -96 0zm51.3 182.7L224.2 307l49.7 49.7c9 9 14.1 21.2 14.1 33.9l0 89.4c0 17.7-14.3 32-32 32s-32-14.3-32-32l0-82.7-73.9-73.9c-15.8-15.8-22.2-38.6-16.9-60.3l20.4-84c8.3-34.1 42.7-54.9 76.7-46.4c19 4.8 35.6 16.4 46.4 32.7L305.1 208l30.9 0 0-24c0-13.3 10.7-24 24-24s24 10.7 24 24l0 55.8c0 .1 0 .2 0 .2s0 .2 0 .2L384 488c0 13.3-10.7 24-24 24s-24-10.7-24-24l0-216-39.4 0c-16 0-31-8-39.9-21.4l-13.3-20zM81.1 471.9L117.3 334c3 4.2 6.4 8.2 10.1 11.9l41.9 41.9L142.9 488.1c-4.5 17.1-22 27.3-39.1 22.8s-27.3-22-22.8-39.1zm55.5-346L101.4 266.5c-3 12.1-14.9 19.9-27.2 17.9l-47.9-8c-14-2.3-22.9-16.3-19.2-30L31.9 155c9.5-34.8 41.1-59 77.2-59l4.2 0c15.6 0 27.1 14.7 23.3 29.8z',
				),
				'fas fa-running'          => array(
					'icon'     => 'fas fa-person-running',
					'view_box' => '0 0 448 512',
					'path'     => 'M320 48a48 48 0 1 0 -96 0 48 48 0 1 0 96 0zM125.7 175.5c9.9-9.9 23.4-15.5 37.5-15.5c1.9 0 3.8 .1 5.6 .3L137.6 254c-9.3 28 1.7 58.8 26.8 74.5l86.2 53.9-25.4 88.8c-4.9 17 5 34.7 22 39.6s34.7-5 39.6-22l28.7-100.4c5.9-20.6-2.6-42.6-20.7-53.9L238 299l30.9-82.4 5.1 12.3C289 264.7 323.9 288 362.7 288l21.3 0c17.7 0 32-14.3 32-32s-14.3-32-32-32l-21.3 0c-12.9 0-24.6-7.8-29.5-19.7l-6.3-15c-14.6-35.1-44.1-61.9-80.5-73.1l-48.7-15c-11.1-3.4-22.7-5.2-34.4-5.2c-31 0-60.8 12.3-82.7 34.3L57.4 153.4c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0l23.1-23.1zM91.2 352L32 352c-17.7 0-32 14.3-32 32s14.3 32 32 32l69.6 0c19 0 36.2-11.2 43.9-28.5L157 361.6l-9.5-6c-17.5-10.9-30.5-26.8-37.9-44.9L91.2 352z',
				),
			);

			$temp_arr = array();
			foreach ( $trashable_infos['fid'] as $fid ) {
				if ( ! is_array( $field_icons[ $fid ] ?? '' ) ) {
					$temp_arr[ $fid ] = array(
						'icon'     => $trashable_arr[ $field_icons[ $fid ] ]['icon'] ?? $field_icons[ $fid ] ?? '',
						'view_box' => $trashable_arr[ $field_icons[ $fid ] ]['view_box'] ?? '',
						'path'     => $trashable_arr[ $field_icons[ $fid ] ]['path'] ?? '',
					);
				} else {
					$temp_arr[ $fid ] = $field_icons[ $fid ];
				}
			}

			unset( $trashable_arr );

			$trashable_infos['field_icon'] = $temp_arr;

			$this->set( 'trip_facts', $trashable_infos );
		}

		return $trashable_infos;
	}

	/**
	 * Get the email settings.
	 *
	 * @return string
	 * @since 6.2.0
	 */
	protected function get_emails() {

		$emails = parent::get( 'email.emails' );

		if ( is_null( $emails ) ) {
			$emails = Options::get( 'admin_email' );
			$this->set( 'email.emails', $emails );
		}

		return $emails;
	}

	/**
	 * Get the admin booking notification subject.
	 *
	 * @return string
	 * @since 6.2.0
	 */
	protected function get_admin_booking_notify_subject() {

		$subject = parent::get( 'email.booking_notification_subject_admin' ) ?? \WTE_Booking_Emails::get_subject( 'order', 'admin' );

		if ( is_null( parent::get( 'email.booking_notification_subject_admin' ) ) ) {
			$this->set( 'email.booking_notification_subject_admin', $subject );
		}

		return $subject;
	}

	/**
	 * Get the admin booking notification template.
	 *
	 * @return string
	 * @since 6.2.0
	 */
	protected function get_admin_booking_notify_template() {

		$template = parent::get( 'email.booking_notification_template_admin' ) ?? \WTE_Booking_Emails::get_template_content( 'order', 'template-emails/booking/notification.php', 'admin' );

		if ( is_null( parent::get( 'email.booking_notification_template_admin' ) ) ) {
			$this->set( 'email.booking_notification_template_admin', $template );
		}

		return $template;
	}

	/**
	 * Get the admin payment notification subject.
	 *
	 * @return string
	 * @since 6.2.0
	 */
	protected function get_admin_payment_notify_subject() {

		$subject = parent::get( 'email.sale_subject' ) ?? \WTE_Booking_Emails::get_subject( 'order_confirmation', 'admin' );

		if ( is_null( parent::get( 'email.sale_subject' ) ) ) {
			$this->set( 'email.sale_subject', $subject );
		}

		return $subject;
	}

	/**
	 * Get the admin payment notification template.
	 *
	 * @return string
	 * @since 6.2.0
	 */
	protected function get_admin_payment_notify_template() {

		$template = parent::get( 'email.sales_wpeditor' ) ?? \WTE_Booking_Emails::get_template_content( 'order_confirmation', 'template-emails/booking/confirmation.php', 'admin' );

		if ( is_null( parent::get( 'email.sales_wpeditor' ) ) ) {
			$this->set( 'email.sales_wpeditor', $template );
		}

		return $template;
	}

	/**
	 * Get the customer booking notification subject.
	 *
	 * @return string
	 * @since 6.2.0
	 */
	protected function get_customer_booking_notify_subject() {

		$subject = parent::get( 'email.booking_notification_subject_customer' ) ?? \WTE_Booking_Emails::get_subject( 'order', 'customer' );

		if ( is_null( parent::get( 'email.booking_notification_subject_customer' ) ) ) {
			$this->set( 'email.booking_notification_subject_customer', $subject );
		}

		return $subject;
	}

	/**
	 * Get the customer booking notification subject.
	 *
	 * @return string
	 * @since 6.2.0
	 */
	protected function get_customer_booking_notify_template() {

		$subject = parent::get( 'email.booking_notification_template_customer' ) ?? \WTE_Booking_Emails::get_template_content( 'order', 'template-emails/booking/notification.php', 'customer' );

		if ( is_null( parent::get( 'email.booking_notification_template_customer' ) ) ) {
			$this->set( 'email.booking_notification_template_customer', $subject );
		}

		return $subject;
	}

	/**
	 * Get the customer purchase notification subject.
	 *
	 * @return string
	 * @since 6.2.0
	 */
	protected function get_customer_purchase_notify_subject() {

		$subject = parent::get( 'email.subject' ) ?? \WTE_Booking_Emails::get_subject( 'order_confirmation', 'customer' );

		if ( is_null( parent::get( 'email.subject' ) ) ) {
			$this->set( 'email.subject', $subject );
		}

		return $subject;
	}

	/**
	 * Get the customer purchase notification template.
	 *
	 * @return string
	 * @since 6.2.0
	 */
	protected function get_customer_purchase_notify_template() {

		$template = parent::get( 'email.purchase_wpeditor' ) ?? \WTE_Booking_Emails::get_template_content( 'order_confirmation', 'template-emails/booking/confirmation.php', 'customer' );

		if ( is_null( parent::get( 'email.purchase_wpeditor' ) ) ) {
			$this->set( 'email.purchase_wpeditor', $template );
		}

		return $template;
	}

	/**
	 * Insert default traveler categories.F
	 *
	 * @return array
	 */
	protected function insert_default_traveler_categories(): array {
		$pricing_taxonomy = 'trip-packages-categories';
		$terms            = array();

		$terms[] = (object) wp_insert_term( 'Adult', $pricing_taxonomy, array( 'slug' => 'adult' ) );
		$terms[] = (object) wp_insert_term( 'Child', $pricing_taxonomy, array( 'slug' => 'child' ) );

		Options::update( 'primary_pricing_category', $terms[0]->term_id );

		return $terms;
	}

	/**
	 * Get traveler categories, creates terms if not exists.
	 *
	 * @return WP_Term[]
	 */
	public function get_traveler_categories( array $args = array() ): array {
		$default = array(
			'taxonomy'   => 'trip-packages-categories',
			'hide_empty' => false,
		);
		$terms   = get_terms( wp_parse_args( $args, $default ) );
		if ( empty( $terms ) ) {
			$terms = array_map(
				function ( $term ) {
					return get_term( $term->term_id, 'trip-packages-categories' );
				},
				$this->insert_default_traveler_categories()
			);
		}

		return is_wp_error( $terms ) ? array() : $terms;
	}

	/**
	 * Get the primary pricing category.
	 *
	 * @return WP_Term
	 */
	public function get_primary_pricing_category(): WP_Term {

		$category = get_term(
			Options::get( 'primary_pricing_category' ),
			'trip-packages-categories'
		);

		if ( $category instanceof WP_Term ) {
			return $category;
		}

		$terms = $this->get_traveler_categories();

		return $terms[0];
	}

	/**
	 * Set the default values.
	 *
	 * @return void
	 */
	private function set_defaults() {
		if ( is_null( Options::get( 'wte_update_mail_template' ) ) ) {
			Options::update(
				'wte_update_mail_template',
				wptravelengine_replace(
					version_compare(
						Options::get( 'wptravelengine_since' ),
						'6.5.0',
						'>='
					),
					true,
					'yes',
					'no'
				)
			);
		}
	}
}
